import React, { useState } from 'react';
import './App.css';

interface API {
  id: string;
  name: string;
  description: string;
  category: string;
  pricing: string;
  latency: string;
  uptime: string;
  requests: string;
  icon: string;
}

function App() {
  const [apis] = useState<API[]>([
    {
      id: '1',
      name: 'Weather API Pro',
      description: 'Real-time weather data for any location worldwide',
      category: 'Weather',
      pricing: '$0.001/request',
      latency: '45ms',
      uptime: '99.99%',
      requests: '2.5M/day',
      icon: '🌤️'
    },
    {
      id: '2',
      name: 'Translation Engine',
      description: 'AI-powered translation for 100+ languages',
      category: 'AI/ML',
      pricing: '$0.002/request',
      latency: '120ms',
      uptime: '99.95%',
      requests: '1.2M/day',
      icon: '🌐'
    },
    {
      id: '3',
      name: 'Payment Gateway',
      description: 'Secure payment processing with global coverage',
      category: 'Finance',
      pricing: '2.9% + $0.30',
      latency: '200ms',
      uptime: '99.999%',
      requests: '500K/day',
      icon: '💳'
    }
  ]);

  const [selectedApi, setSelectedApi] = useState<API | null>(null);

  const codeExample = `const response = await fetch('https://api.example.com/v1/endpoint', {
  method: 'GET',
  headers: {
    'Authorization': 'Bearer YOUR_API_KEY',
    'Content-Type': 'application/json'
  }
});

const data = await response.json();
console.log(data);`;

  return (
    <div className="app">
      <header className="header">
        <h1>API Marketplace</h1>
        <nav>
          <button className="nav-btn active">Browse APIs</button>
          <button className="nav-btn">My APIs</button>
          <button className="nav-btn">Analytics</button>
          <button className="nav-btn">Documentation</button>
        </nav>
        <button className="publish-btn">Publish API</button>
      </header>

      <div className="container">
        <aside className="filters">
          <h3>Categories</h3>
          <ul>
            <li className="filter-item active">All APIs</li>
            <li className="filter-item">AI/ML</li>
            <li className="filter-item">Finance</li>
            <li className="filter-item">Weather</li>
            <li className="filter-item">Social Media</li>
            <li className="filter-item">Analytics</li>
            <li className="filter-item">Communication</li>
          </ul>

          <h3>Pricing</h3>
          <ul>
            <li className="filter-item">Free</li>
            <li className="filter-item">Freemium</li>
            <li className="filter-item">Pay per use</li>
            <li className="filter-item">Monthly subscription</li>
          </ul>
        </aside>

        <main className="main-content">
          <div className="search-bar">
            <input type="text" placeholder="Search APIs..." />
            <button className="search-btn">Search</button>
          </div>

          <div className="api-grid">
            {apis.map(api => (
              <div key={api.id} className="api-card" onClick={() => setSelectedApi(api)}>
                <div className="api-header">
                  <span className="api-icon">{api.icon}</span>
                  <div>
                    <h3>{api.name}</h3>
                    <p className="api-category">{api.category}</p>
                  </div>
                </div>
                <p className="api-description">{api.description}</p>
                <div className="api-stats">
                  <div className="stat">
                    <span className="stat-label">Latency</span>
                    <span className="stat-value">{api.latency}</span>
                  </div>
                  <div className="stat">
                    <span className="stat-label">Uptime</span>
                    <span className="stat-value">{api.uptime}</span>
                  </div>
                  <div className="stat">
                    <span className="stat-label">Requests</span>
                    <span className="stat-value">{api.requests}</span>
                  </div>
                </div>
                <div className="api-footer">
                  <span className="pricing">{api.pricing}</span>
                  <button className="try-btn">Try it</button>
                </div>
              </div>
            ))}
          </div>
        </main>

        {selectedApi && (
          <aside className="api-details">
            <div className="details-header">
              <h2>{selectedApi.name}</h2>
              <button onClick={() => setSelectedApi(null)}>×</button>
            </div>
            
            <div className="details-content">
              <h3>Quick Start</h3>
              <div className="code-block">
                <pre><code>{codeExample}</code></pre>
              </div>

              <h3>Endpoints</h3>
              <ul className="endpoints">
                <li>
                  <span className="method get">GET</span>
                  <code>/v1/data</code>
                </li>
                <li>
                  <span className="method post">POST</span>
                  <code>/v1/process</code>
                </li>
                <li>
                  <span className="method put">PUT</span>
                  <code>/v1/update/:id</code>
                </li>
                <li>
                  <span className="method delete">DELETE</span>
                  <code>/v1/delete/:id</code>
                </li>
              </ul>

              <h3>Pricing Plans</h3>
              <div className="pricing-plans">
                <div className="plan">
                  <h4>Free Tier</h4>
                  <p>1,000 requests/month</p>
                </div>
                <div className="plan">
                  <h4>Pro</h4>
                  <p>$49/month</p>
                  <p>100,000 requests</p>
                </div>
                <div className="plan">
                  <h4>Enterprise</h4>
                  <p>Custom pricing</p>
                </div>
              </div>

              <button className="subscribe-btn">Subscribe to API</button>
            </div>
          </aside>
        )}
      </div>
    </div>
  );
}

export default App;