import React, { useState } from 'react';
import BlogEditor from './components/BlogEditor';
import BlogList from './components/BlogList';
import BlogPost from './components/BlogPost';
import './App.css';

export interface Post {
  id: string;
  title: string;
  content: string;
  excerpt: string;
  author: string;
  date: string;
  tags: string[];
  published: boolean;
}

function App() {
  const [view, setView] = useState<'list' | 'editor' | 'post'>('list');
  const [posts, setPosts] = useState<Post[]>([
    {
      id: '1',
      title: 'Getting Started with React',
      content: '# Getting Started with React\n\nReact is a powerful library for building user interfaces...',
      excerpt: 'Learn the basics of React and start building amazing applications.',
      author: 'John Doe',
      date: new Date().toISOString(),
      tags: ['react', 'javascript', 'tutorial'],
      published: true
    }
  ]);
  const [selectedPost, setSelectedPost] = useState<Post | null>(null);

  const handleSavePost = (post: Post) => {
    if (post.id) {
      setPosts(posts.map(p => p.id === post.id ? post : p));
    } else {
      setPosts([...posts, { ...post, id: Date.now().toString() }]);
    }
    setView('list');
  };

  const handleEditPost = (post: Post) => {
    setSelectedPost(post);
    setView('editor');
  };

  const handleViewPost = (post: Post) => {
    setSelectedPost(post);
    setView('post');
  };

  return (
    <div className="app">
      <header className="header">
        <div className="header-content">
          <h1>Blog Editor</h1>
          <nav>
            <button onClick={() => setView('list')} className={`nav-btn ${view === 'list' ? 'active' : ''}`}>
              Posts
            </button>
            <button onClick={() => { setSelectedPost(null); setView('editor'); }} className={`nav-btn ${view === 'editor' ? 'active' : ''}`}>
              New Post
            </button>
          </nav>
        </div>
      </header>

      <main className="main">
        {view === 'list' && (
          <BlogList posts={posts} onEdit={handleEditPost} onView={handleViewPost} />
        )}
        {view === 'editor' && (
          <BlogEditor post={selectedPost} onSave={handleSavePost} onCancel={() => setView('list')} />
        )}
        {view === 'post' && selectedPost && (
          <BlogPost post={selectedPost} onEdit={() => handleEditPost(selectedPost)} onBack={() => setView('list')} />
        )}
      </main>
    </div>
  );
}

export default App;