import React, { useState } from 'react';
import MDEditor from '@uiw/react-md-editor';
import { Post } from '../App';
import './BlogEditor.css';

interface BlogEditorProps {
  post: Post | null;
  onSave: (post: Post) => void;
  onCancel: () => void;
}

export default function BlogEditor({ post, onSave, onCancel }: BlogEditorProps) {
  const [title, setTitle] = useState(post?.title || '');
  const [content, setContent] = useState(post?.content || '');
  const [excerpt, setExcerpt] = useState(post?.excerpt || '');
  const [author, setAuthor] = useState(post?.author || 'John Doe');
  const [tags, setTags] = useState(post?.tags.join(', ') || '');
  const [published, setPublished] = useState(post?.published || false);

  const handleSave = () => {
    const newPost: Post = {
      id: post?.id || '',
      title,
      content,
      excerpt,
      author,
      date: post?.date || new Date().toISOString(),
      tags: tags.split(',').map(tag => tag.trim()).filter(Boolean),
      published
    };
    onSave(newPost);
  };

  return (
    <div className="blog-editor">
      <div className="editor-header">
        <h2>{post ? 'Edit Post' : 'New Post'}</h2>
        <div className="editor-actions">
          <button onClick={onCancel} className="btn btn-secondary">Cancel</button>
          <button onClick={handleSave} className="btn btn-primary">Save</button>
        </div>
      </div>

      <div className="editor-form">
        <div className="form-group">
          <label>Title</label>
          <input
            type="text"
            value={title}
            onChange={(e) => setTitle(e.target.value)}
            placeholder="Enter post title..."
          />
        </div>

        <div className="form-group">
          <label>Excerpt</label>
          <textarea
            value={excerpt}
            onChange={(e) => setExcerpt(e.target.value)}
            placeholder="Brief description of your post..."
            rows={3}
          />
        </div>

        <div className="form-row">
          <div className="form-group">
            <label>Author</label>
            <input
              type="text"
              value={author}
              onChange={(e) => setAuthor(e.target.value)}
            />
          </div>

          <div className="form-group">
            <label>Tags (comma separated)</label>
            <input
              type="text"
              value={tags}
              onChange={(e) => setTags(e.target.value)}
              placeholder="react, javascript, tutorial"
            />
          </div>

          <div className="form-group">
            <label className="checkbox-label">
              <input
                type="checkbox"
                checked={published}
                onChange={(e) => setPublished(e.target.checked)}
              />
              <span>Publish immediately</span>
            </label>
          </div>
        </div>

        <div className="form-group">
          <label>Content</label>
          <MDEditor
            value={content}
            onChange={(val) => setContent(val || '')}
            preview="edit"
            height={400}
          />
        </div>
      </div>
    </div>
  );
}