import React, { useState } from 'react';
import Sidebar from './components/Sidebar';
import ContentList from './components/ContentList';
import ContentEditor from './components/ContentEditor';
import MediaLibrary from './components/MediaLibrary';
import './App.css';

export interface Content {
  id: string;
  title: string;
  content: string;
  type: 'page' | 'post' | 'product';
  status: 'draft' | 'published' | 'scheduled';
  author: string;
  createdAt: string;
  updatedAt: string;
  site: string;
  slug: string;
  seo: {
    title: string;
    description: string;
    keywords: string;
  };
}

export interface Site {
  id: string;
  name: string;
  domain: string;
  theme: string;
}

function App() {
  const [view, setView] = useState<'content' | 'editor' | 'media' | 'settings'>('content');
  const [selectedSite, setSelectedSite] = useState<string>('site-1');
  const [selectedContent, setSelectedContent] = useState<Content | null>(null);
  
  const [sites] = useState<Site[]>([
    { id: 'site-1', name: 'Main Website', domain: 'example.com', theme: 'default' },
    { id: 'site-2', name: 'Blog', domain: 'blog.example.com', theme: 'minimal' }
  ]);

  const [contents, setContents] = useState<Content[]>([
    {
      id: '1',
      title: 'Welcome to Our CMS',
      content: '<h1>Welcome</h1><p>This is your content management system.</p>',
      type: 'page',
      status: 'published',
      author: 'Admin',
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString(),
      site: 'site-1',
      slug: 'welcome',
      seo: {
        title: 'Welcome - CMS',
        description: 'Welcome to our content management system',
        keywords: 'cms, content, management'
      }
    }
  ]);

  const handleSaveContent = (content: Content) => {
    if (content.id) {
      setContents(contents.map(c => c.id === content.id ? content : c));
    } else {
      setContents([...contents, { ...content, id: Date.now().toString() }]);
    }
    setSelectedContent(null);
    setView('content');
  };

  const handleEditContent = (content: Content) => {
    setSelectedContent(content);
    setView('editor');
  };

  const handleDeleteContent = (id: string) => {
    setContents(contents.filter(c => c.id !== id));
  };

  const filteredContents = contents.filter(c => c.site === selectedSite);

  return (
    <div className="app">
      <Sidebar
        view={view}
        onViewChange={setView}
        sites={sites}
        selectedSite={selectedSite}
        onSiteChange={setSelectedSite}
        onNewContent={() => {
          setSelectedContent(null);
          setView('editor');
        }}
      />
      
      <main className="main">
        {view === 'content' && (
          <ContentList
            contents={filteredContents}
            onEdit={handleEditContent}
            onDelete={handleDeleteContent}
          />
        )}
        
        {view === 'editor' && (
          <ContentEditor
            content={selectedContent}
            site={selectedSite}
            onSave={handleSaveContent}
            onCancel={() => {
              setSelectedContent(null);
              setView('content');
            }}
          />
        )}
        
        {view === 'media' && <MediaLibrary />}
        
        {view === 'settings' && (
          <div className="settings">
            <h2>Settings</h2>
            <p>Site configuration and settings will appear here.</p>
          </div>
        )}
      </main>
    </div>
  );
}

export default App;