import React, { useState } from 'react';
import { Editor } from '@tiptap/react';
import { useEditor, EditorContent } from '@tiptap/react';
import StarterKit from '@tiptap/starter-kit';
import Link from '@tiptap/extension-link';
import Image from '@tiptap/extension-image';
import { Content } from '../App';
import './ContentEditor.css';

interface ContentEditorProps {
  content: Content | null;
  site: string;
  onSave: (content: Content) => void;
  onCancel: () => void;
}

export default function ContentEditor({ content, site, onSave, onCancel }: ContentEditorProps) {
  const [title, setTitle] = useState(content?.title || '');
  const [slug, setSlug] = useState(content?.slug || '');
  const [type, setType] = useState<'page' | 'post' | 'product'>(content?.type || 'page');
  const [status, setStatus] = useState<'draft' | 'published' | 'scheduled'>(content?.status || 'draft');
  const [seoTitle, setSeoTitle] = useState(content?.seo.title || '');
  const [seoDescription, setSeoDescription] = useState(content?.seo.description || '');
  const [seoKeywords, setSeoKeywords] = useState(content?.seo.keywords || '');

  const editor = useEditor({
    extensions: [
      StarterKit,
      Link.configure({
        openOnClick: false,
      }),
      Image
    ],
    content: content?.content || '<p>Start writing your content here...</p>',
  });

  const handleSave = () => {
    if (!editor) return;

    const newContent: Content = {
      id: content?.id || '',
      title,
      content: editor.getHTML(),
      type,
      status,
      author: 'Admin',
      createdAt: content?.createdAt || new Date().toISOString(),
      updatedAt: new Date().toISOString(),
      site,
      slug: slug || title.toLowerCase().replace(/\s+/g, '-'),
      seo: {
        title: seoTitle || title,
        description: seoDescription,
        keywords: seoKeywords
      }
    };

    onSave(newContent);
  };

  const MenuBar = ({ editor }: { editor: Editor }) => {
    if (!editor) return null;

    return (
      <div className="menu-bar">
        <button
          onClick={() => editor.chain().focus().toggleBold().run()}
          className={editor.isActive('bold') ? 'is-active' : ''}
        >
          B
        </button>
        <button
          onClick={() => editor.chain().focus().toggleItalic().run()}
          className={editor.isActive('italic') ? 'is-active' : ''}
        >
          I
        </button>
        <button
          onClick={() => editor.chain().focus().toggleHeading({ level: 1 }).run()}
          className={editor.isActive('heading', { level: 1 }) ? 'is-active' : ''}
        >
          H1
        </button>
        <button
          onClick={() => editor.chain().focus().toggleHeading({ level: 2 }).run()}
          className={editor.isActive('heading', { level: 2 }) ? 'is-active' : ''}
        >
          H2
        </button>
        <button
          onClick={() => editor.chain().focus().toggleBulletList().run()}
          className={editor.isActive('bulletList') ? 'is-active' : ''}
        >
          List
        </button>
        <button
          onClick={() => {
            const url = window.prompt('Enter URL:');
            if (url) {
              editor.chain().focus().setLink({ href: url }).run();
            }
          }}
          className={editor.isActive('link') ? 'is-active' : ''}
        >
          Link
        </button>
      </div>
    );
  };

  return (
    <div className="content-editor">
      <div className="editor-header">
        <h2>{content ? 'Edit Content' : 'New Content'}</h2>
        <div className="editor-actions">
          <button onClick={onCancel} className="btn btn-secondary">Cancel</button>
          <button onClick={handleSave} className="btn btn-primary">Save</button>
        </div>
      </div>

      <div className="editor-form">
        <div className="form-section">
          <h3>Content Details</h3>
          <div className="form-grid">
            <div className="form-group">
              <label>Title</label>
              <input
                type="text"
                value={title}
                onChange={(e) => setTitle(e.target.value)}
                placeholder="Enter title..."
              />
            </div>
            <div className="form-group">
              <label>URL Slug</label>
              <input
                type="text"
                value={slug}
                onChange={(e) => setSlug(e.target.value)}
                placeholder="url-slug"
              />
            </div>
            <div className="form-group">
              <label>Type</label>
              <select value={type} onChange={(e) => setType(e.target.value as any)}>
                <option value="page">Page</option>
                <option value="post">Post</option>
                <option value="product">Product</option>
              </select>
            </div>
            <div className="form-group">
              <label>Status</label>
              <select value={status} onChange={(e) => setStatus(e.target.value as any)}>
                <option value="draft">Draft</option>
                <option value="published">Published</option>
                <option value="scheduled">Scheduled</option>
              </select>
            </div>
          </div>
        </div>

        <div className="form-section">
          <h3>Content</h3>
          <div className="editor-wrapper">
            {editor && <MenuBar editor={editor} />}
            <EditorContent editor={editor} className="editor-content" />
          </div>
        </div>

        <div className="form-section">
          <h3>SEO Settings</h3>
          <div className="form-group">
            <label>SEO Title</label>
            <input
              type="text"
              value={seoTitle}
              onChange={(e) => setSeoTitle(e.target.value)}
              placeholder="SEO optimized title..."
            />
          </div>
          <div className="form-group">
            <label>Meta Description</label>
            <textarea
              value={seoDescription}
              onChange={(e) => setSeoDescription(e.target.value)}
              placeholder="Brief description for search engines..."
              rows={3}
            />
          </div>
          <div className="form-group">
            <label>Keywords</label>
            <input
              type="text"
              value={seoKeywords}
              onChange={(e) => setSeoKeywords(e.target.value)}
              placeholder="keyword1, keyword2, keyword3"
            />
          </div>
        </div>
      </div>
    </div>
  );
}