import React from 'react';
import { Content } from '../App';
import './ContentList.css';

interface ContentListProps {
  contents: Content[];
  onEdit: (content: Content) => void;
  onDelete: (id: string) => void;
}

export default function ContentList({ contents, onEdit, onDelete }: ContentListProps) {
  const formatDate = (date: string) => {
    return new Date(date).toLocaleDateString('en-US', {
      month: 'short',
      day: 'numeric',
      year: 'numeric'
    });
  };

  const getTypeIcon = (type: string) => {
    switch (type) {
      case 'page': return '📄';
      case 'post': return '📝';
      case 'product': return '🛍️';
      default: return '📄';
    }
  };

  return (
    <div className="content-list">
      <div className="content-header">
        <h2>Content</h2>
        <div className="content-filters">
          <select className="filter-select">
            <option>All Types</option>
            <option>Pages</option>
            <option>Posts</option>
            <option>Products</option>
          </select>
          <select className="filter-select">
            <option>All Statuses</option>
            <option>Published</option>
            <option>Draft</option>
            <option>Scheduled</option>
          </select>
        </div>
      </div>

      <div className="content-table">
        <table>
          <thead>
            <tr>
              <th>Title</th>
              <th>Type</th>
              <th>Status</th>
              <th>Author</th>
              <th>Updated</th>
              <th>Actions</th>
            </tr>
          </thead>
          <tbody>
            {contents.map(content => (
              <tr key={content.id}>
                <td>
                  <div className="content-title">
                    <span className="type-icon">{getTypeIcon(content.type)}</span>
                    <div>
                      <div className="title">{content.title}</div>
                      <div className="slug">/{content.slug}</div>
                    </div>
                  </div>
                </td>
                <td>
                  <span className="content-type">{content.type}</span>
                </td>
                <td>
                  <span className={`status ${content.status}`}>
                    {content.status}
                  </span>
                </td>
                <td>{content.author}</td>
                <td>{formatDate(content.updatedAt)}</td>
                <td>
                  <div className="actions">
                    <button onClick={() => onEdit(content)} className="btn-icon" title="Edit">
                      ✏️
                    </button>
                    <button onClick={() => window.open(`/${content.slug}`, '_blank')} className="btn-icon" title="View">
                      👁️
                    </button>
                    <button onClick={() => onDelete(content.id)} className="btn-icon" title="Delete">
                      🗑️
                    </button>
                  </div>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
    </div>
  );
}