import React, { useState } from 'react'
import { Navigation } from './components/Navigation'
import { ProductsPage } from './components/ProductsPage'
import { CartPage } from './components/CartPage'

interface CartItem {
  id: string
  name: string
  price: number
  quantity: number
  icon: string
}

export default function App() {
  const [currentPage, setCurrentPage] = useState('products')
  const [cartItems, setCartItems] = useState<CartItem[]>([])
  const [config] = useState({
    appName: 'Store',
    primaryColor: '#2563eb',
    storeName: 'My Store',
    currency: 'USD'
  })

  const navigation = [
    { name: 'Products', href: '#products', current: currentPage === 'products' },
    { name: `Cart (${cartItems.length})`, href: '#cart', current: currentPage === 'cart' },
    { name: 'Orders', href: '#orders', current: currentPage === 'orders' },
    { name: 'Account', href: '#account', current: currentPage === 'account' },
  ]

  const handlePageChange = (page: string) => {
    setCurrentPage(page.toLowerCase().split(' ')[0])
  }

  const addToCart = (product: { id: string; name: string; price: number; icon: string }) => {
    setCartItems(prev => {
      const existing = prev.find(item => item.id === product.id)
      if (existing) {
        return prev.map(item =>
          item.id === product.id
            ? { ...item, quantity: item.quantity + 1 }
            : item
        )
      }
      return [...prev, { ...product, quantity: 1 }]
    })
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <Navigation 
        appName={config.storeName}
        pages={navigation}
        primaryColor={config.primaryColor}
        onPageChange={handlePageChange}
      />
      
      <main className="pt-16">
        {currentPage === 'products' && (
          <ProductsPage config={config} onAddToCart={addToCart} />
        )}
        {currentPage === 'cart' && (
          <CartPage config={config} items={cartItems} onUpdateItems={setCartItems} />
        )}
        {currentPage === 'orders' && (
          <div className="max-w-7xl mx-auto px-4 py-8">
            <h1 className="text-3xl font-bold">Orders</h1>
            <p className="mt-4">Order history coming soon...</p>
          </div>
        )}
        {currentPage === 'account' && (
          <div className="max-w-7xl mx-auto px-4 py-8">
            <h1 className="text-3xl font-bold">Account</h1>
            <p className="mt-4">Account settings coming soon...</p>
          </div>
        )}
      </main>
    </div>
  )
}