import React, { useState } from 'react';
import './App.css';

interface Task {
  id: string;
  title: string;
  completed: boolean;
  priority: 'low' | 'medium' | 'high';
  dueDate?: string;
  project?: string;
}

function App() {
  const [tasks, setTasks] = useState<Task[]>([
    {
      id: '1',
      title: 'Complete project proposal',
      completed: false,
      priority: 'high',
      dueDate: 'Today',
      project: 'Work'
    },
    {
      id: '2',
      title: 'Review design mockups',
      completed: true,
      priority: 'medium',
      dueDate: 'Yesterday',
      project: 'Design'
    },
    {
      id: '3',
      title: 'Team meeting preparation',
      completed: false,
      priority: 'medium',
      dueDate: 'Tomorrow',
      project: 'Work'
    }
  ]);

  const [newTask, setNewTask] = useState('');
  const [view, setView] = useState<'list' | 'kanban'>('list');

  const addTask = () => {
    if (newTask.trim()) {
      const task: Task = {
        id: Date.now().toString(),
        title: newTask,
        completed: false,
        priority: 'medium',
        dueDate: 'Today'
      };
      setTasks([...tasks, task]);
      setNewTask('');
    }
  };

  const toggleTask = (id: string) => {
    setTasks(tasks.map(task =>
      task.id === id ? { ...task, completed: !task.completed } : task
    ));
  };

  const deleteTask = (id: string) => {
    setTasks(tasks.filter(task => task.id !== id));
  };

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case 'high': return 'priority-high';
      case 'medium': return 'priority-medium';
      case 'low': return 'priority-low';
      default: return '';
    }
  };

  return (
    <div className="app">
      <header className="header">
        <h1>Productivity Hub</h1>
        <div className="header-actions">
          <button className="view-toggle" onClick={() => setView(view === 'list' ? 'kanban' : 'list')}>
            {view === 'list' ? '📋 List View' : '📊 Kanban View'}
          </button>
        </div>
      </header>

      <div className="container">
        <aside className="sidebar">
          <nav className="nav">
            <button className="nav-item active">
              <span>📝</span> All Tasks
            </button>
            <button className="nav-item">
              <span>📅</span> Today
            </button>
            <button className="nav-item">
              <span>📆</span> This Week
            </button>
            <button className="nav-item">
              <span>🔴</span> High Priority
            </button>
          </nav>

          <div className="projects">
            <h3>Projects</h3>
            <ul>
              <li className="project-item">
                <span className="project-color" style={{ background: '#3b82f6' }}></span>
                Work
              </li>
              <li className="project-item">
                <span className="project-color" style={{ background: '#10b981' }}></span>
                Personal
              </li>
              <li className="project-item">
                <span className="project-color" style={{ background: '#f59e0b' }}></span>
                Design
              </li>
            </ul>
          </div>
        </aside>

        <main className="main">
          <div className="task-input">
            <input
              type="text"
              value={newTask}
              onChange={(e) => setNewTask(e.target.value)}
              onKeyPress={(e) => e.key === 'Enter' && addTask()}
              placeholder="Add a new task..."
            />
            <button onClick={addTask} className="add-btn">Add Task</button>
          </div>

          {view === 'list' ? (
            <div className="task-list">
              {tasks.map(task => (
                <div key={task.id} className={`task-item ${task.completed ? 'completed' : ''}`}>
                  <input
                    type="checkbox"
                    checked={task.completed}
                    onChange={() => toggleTask(task.id)}
                  />
                  <div className="task-content">
                    <h4>{task.title}</h4>
                    <div className="task-meta">
                      <span className={`priority ${getPriorityColor(task.priority)}`}>
                        {task.priority}
                      </span>
                      {task.dueDate && <span className="due-date">📅 {task.dueDate}</span>}
                      {task.project && <span className="project">🏷️ {task.project}</span>}
                    </div>
                  </div>
                  <button onClick={() => deleteTask(task.id)} className="delete-btn">×</button>
                </div>
              ))}
            </div>
          ) : (
            <div className="kanban-board">
              <div className="kanban-column">
                <h3>To Do</h3>
                <div className="kanban-tasks">
                  {tasks.filter(t => !t.completed).map(task => (
                    <div key={task.id} className="kanban-card">
                      <h4>{task.title}</h4>
                      <div className="task-meta">
                        <span className={`priority ${getPriorityColor(task.priority)}`}>
                          {task.priority}
                        </span>
                        {task.dueDate && <span className="due-date">{task.dueDate}</span>}
                      </div>
                    </div>
                  ))}
                </div>
              </div>
              <div className="kanban-column">
                <h3>Completed</h3>
                <div className="kanban-tasks">
                  {tasks.filter(t => t.completed).map(task => (
                    <div key={task.id} className="kanban-card completed">
                      <h4>{task.title}</h4>
                      <div className="task-meta">
                        <span className={`priority ${getPriorityColor(task.priority)}`}>
                          {task.priority}
                        </span>
                        {task.dueDate && <span className="due-date">{task.dueDate}</span>}
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            </div>
          )}
        </main>
      </div>
    </div>
  );
}

export default App;